function [ShiftedSpectrum] = alignspectra(Std1Low,Std2High,ToBeShifted1Low,ToBeShifted2High,ToBeShiftedSpectra)

% This program aligns a mass spectrum such that peaks will occur at
% specified m/z values.  This is done in an interpolation manner.  Two
% peaks are chosen by a user.  First, the spectrum is shifted to align
% itself with the first/early peak.  Second, the spectrum is
% stretched/shrunk such that the later m/z values line up.  This work is
% based on X. F. Li, H. J. Ren, X. C. Le, M. Qi, I. D. Ireland and N. J. 
% Dovichi, J Chromatogr A, 2000, 869, 375-384. 
%
%INPUTS:
%ToBeShiftedSpectra - The mass spectrum that needs to be shifted/aligned
%Std1Low - m/z value of an early peak in the spectrum that you wish to
%align to
%Std2High - m/z value of a later peak in the spectrum that you wish to
%align to
%ToBeShifted1Low - m/z value of peak in a spectrum that needs to be
%shifted/aligned
%ToBeShifted2High - m/z value of peak in a spectrum that needs to be
%shifted/aligned
%
%OUTPUT:
%ShiftedSpectrum - Shifted mass spectrum, aligned around m/z values of
%interest

mz=1:length(ToBeShiftedSpectra);%mz axis

DeltamzStd=Std2High-Std1Low;
DeltamzShift=ToBeShifted2High-ToBeShifted1Low;

mzCorrected=(mz-ToBeShifted1Low)*DeltamzStd/DeltamzShift+Std1Low;
ShiftedSpectrum=interp1(mzCorrected(mz),ToBeShiftedSpectra(mz),mz);

%
%set NaNs to zero
%
AllNaNs=find(isnan(ShiftedSpectrum)==1);
ShiftedSpectrum(AllNaNs)=zeros(size(AllNaNs));